#version 150

/**
 * © Janus Bager Kristensen & Rolf Bagge, CAVI Aarhus University, 2014
 */

/**
 * Return a grayscale image with white signifying points that have a
 * bullseye tracking circle centered around them and black being points that
 * do not.
 */

uniform sampler2D texture;

const float radius = %RADIUS%;
const float ringSize = %RING_SIZE%;

const float sensitivity = %SENSITIVITY%;
const float factor = %FILTER%;

const int circleSteps = 32; // The numer of samples to take
const float pi = 3.1415926535;
const float piStep = (2.0 * pi) / circleSteps;
const float minThreshold = %MIN_CENTER_BRIGHTNESS%; // Circles with centers darker than this will not be included
const float rescaler = 5.0; // The input ranges from 0-1 and will be 0 and 1 in a perfect world. In the real world it won't so we scale the output slightly up to better capture that the output will never reach 1.

smooth in vec2 texCoord;
out vec4 fragColor;

void main() {
    vec2 uv = texCoord;
    float original = texture2D(texture,uv).r;

    vec2 textureSize = textureSize(texture, 0);
    vec2 textureUnits = 1.0 / textureSize;

    if(original < minThreshold) {
        fragColor = vec4(0);
        return;
    } else {
        float outsideRadius = radius + ringSize / 2.0; 
        float outermostRadius = radius - ringSize / 2.0; 
        float innermostRadius = ringSize; // The innermost ring is black (the center is white)

        // Find the circle likeliness by sampling the min(diff between outside and outer white circle, diff between center and innermost circle)
        float totalColor = 0;
        float totalCenterDifference = 0;
        float previousCenterColor = texture2D(texture, uv + innermostRadius * textureUnits).r;
        float previousOutsideColor = texture2D(texture, uv + (radius + ringSize*1.5) * textureUnits).r;
        for(int i = 0; i<circleSteps; i++) {
            float radians = i * piStep;
            vec2 directionVector = vec2(cos(radians) * textureUnits.x, sin(radians) * textureUnits.y);

            vec2 outsideCircleUV = uv + outsideRadius * directionVector; // outside
            vec2 outermostCircleUV = uv + outermostRadius * directionVector; // outermost ring
            vec2 innermostCircleUV = uv + innermostRadius * directionVector; // innermost ring

            float outsideColor = texture2D(texture, outsideCircleUV).r; // The outside of the tag should be black
            float outermostColor = texture2D(texture, outermostCircleUV).r; // The outermost ring should be white
            float innermostColor = texture2D(texture, innermostCircleUV).r; //The inner ring should be black

            totalColor += min(outermostColor - outsideColor, original - innermostColor);
            totalCenterDifference += abs(innermostColor - previousCenterColor)+abs(outsideColor-previousOutsideColor);
            previousCenterColor = innermostColor;
            previousOutsideColor = outsideColor;
        }
        totalColor /= circleSteps;
        totalCenterDifference /= circleSteps;

        fragColor = vec4(max(totalColor,0) * sensitivity, 0, totalCenterDifference * factor, 1);
    }
}
